////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.util.UUID;

import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.ClientDisconnectHandler;
import com.crankuptheamps.client.Message;
import com.crankuptheamps.client.MessageHandler;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.exception.ConnectionException;

public class SubscribeCommand extends SparkCommand
{
    String uriString                = null;
    String topic                    = null;
    String filter                   = null;

    public SubscribeCommand(String name)
    {
        super(name);

        // These are the arguments this command recognizes
        addOption(new CommandOption("topic", "topic to query", true));
        addOption(new CommandOption("filter", "content filter to use", false));
        addOption(new CommandOption("delta", "use delta subscription", false));
        addOption(new CommandOption("bookmark", "begin subscription from bookmark", false));
        addOption(new CommandOption("limit", "limit number of mesasages received", false));
        addOption(new CommandOption("timeout", "timeout to wait for command to return in milliseconds", false));
        
        // These are the examples...
        StringBuilder example = new StringBuilder(1024);
        example.append("  % ./spark subscribe -proto fix -server localhost:9003 -topic Trades -filter \"/55 = 'Transport.123'\"\n");
        example.append("    Connects to the AMPS instance listening on port 9003");
        example.append("    using the FIX protocol and prints incoming messages in topic 'Trades'");
        example.append("    that have tag 55 set to 'Transport.123'");
        addExample(example.toString());

    }

    private class SubscribeCommandHandler implements MessageHandler, ClientDisconnectHandler
    {
        int limit = 0, current =0, timeout = 5000;
        public SubscribeCommandHandler(int limit, int timeout)
        {
            this.limit = limit;
            this.timeout = timeout;
        }
        public void invoke(Message message)
        {
            System.out.println(message.getData());
            if(limit>0 && ++ this.current >= limit)
            {
                System.exit(0);
            }
        }
        public void invoke(Client client)
        {
            int attempts = 0;
            while(true)
            {
                if(++attempts > 5) break;
                try
                {
                    System.err.println("DISCONNECTED, trying to reconnect (attempt #" + attempts + ")");
                    client.connect(uriString);
                    client.logon(5000,createAuthenticator(uriString));

                    // Replace subscription
                    client.subscribe(this, topic, filter, timeout);

                    // Success, let's return
                    return;
                }
                catch(AMPSException e)
                {
                    System.err.println("Unable to re-connect to AMPS (" + e + ").");
                    // Cleanup connection for next attempt
                    client.disconnect();
                }

                // Sleep for 1 second before retrying to connect
                try
                {
                    Thread.sleep(1000);
                }
                catch(InterruptedException e)
                {
                    ;
                }

            }
            // We've failed to reconnect, let's notify the main thread that we failed.
            System.err.println("Unable to reestablish connection to AMPS.");
            synchronized(this)
            {
                this.notifyAll();
            }
        }
    }

    void execute(String argv[])
    {
        // Extract all of the command line arguments
        parseArgs(argv, 1);
        String[] userAndServer = options.get("server").getValue().split("@",2);
        String server = (userAndServer.length == 2) ? userAndServer[1] : userAndServer[0];
        topic             = options.get("topic").getValue();
        String type       = options.get("proto").getValue();
        boolean delta     = options.get("delta").hasValue();
        int limit = 0;
        int timeout = 5000;
        
        if(options.get("limit").hasValue())
        {
            limit = Integer.parseInt(options.get("limit").getValue());
        }
        
        
        if(!type.equals("xml") && !type.equals("fix") && !type.equals("nvfix") && !type.equals("amps") && !type.equals("json"))
        {
            badUsage("prot argument needs to be 'fix', 'xml', 'nvfix', 'amps', or 'json'.");
        }

        if(options.get("filter").hasValue())
        {
            filter = options.get("filter").getValue();
        }

        if(options.get("timeout").hasValue())
        {
            timeout = Integer.parseInt(options.get("timeout").getValue());
        }

        // Connect client to AMPS
        String userId = (userAndServer.length == 2) ? userAndServer[0] : System.getProperty("user.name");
        String clientName = "spark-subscribe-" + userId + "-" + UUID.randomUUID().toString();
        Client client = new Client(clientName, Client.Version.AMPS_2);
        SubscribeCommandHandler handler = new SubscribeCommandHandler(limit, timeout);
        client.setDisconnectHandler(handler);
        try
        {
            uriString = "tcp://" + userId + "@" + server + "/" + type;
            client.connect(uriString);
            client.logon(5000,createAuthenticator(uriString));
        }
        catch (ConnectionException e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }
        catch (Exception e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }

        try
        {
            synchronized(handler)
            {
                // Execute the subscribe request, but only wait 5 seconds for it to be
                //   issued, otherwise we abandon it.
                if(delta)
                {
                    client.deltaSubscribe(handler, topic, filter, timeout);
                }
                else
                {
                    if(options.get("bookmark").hasValue())
                    {
                        client.bookmarkSubscribe(handler, topic, filter, null, options.get("bookmark").getValue(),Message.Options.None, timeout);
                    }
                    else
                    {
                        client.subscribe(handler, topic, filter, timeout);
                    }
                }
                try
                {
                    handler.wait();  // This just hangs here until we're killed.
                }
                catch (InterruptedException e)
                {
                    // Nothing to do...
                }
            }
        }
        catch (Exception e)
        {
            // Failed to send
            System.err.println("Error sending subscribe request: " + e.toString());
        }

        client.close();
    }
}
