////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.lang.reflect.Constructor;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.LinkedList;

import com.crankuptheamps.client.Authenticator;
import com.crankuptheamps.client.DefaultAuthenticator;
import com.crankuptheamps.client.exception.AuthenticationException;

public abstract class SparkCommand
{
    protected String                         name;
    protected TreeMap<String, CommandOption> options  = new TreeMap<String, CommandOption>();
    protected TreeSet<String>                required = new TreeSet<String>();
    protected TreeSet<String>                optional = new TreeSet<String>();

    protected LinkedList<String>             examples = new LinkedList<String>();
    protected LinkedList<String>             extras   = new LinkedList<String>();
    protected String                         usage    = null;
    protected TreeMap<String, String>        synonyms = new TreeMap<String, String>();
    
    public SparkCommand(String name)
    {
        this.name = name;
        addOption(new CommandOption("server", "AMPS server to connect to", true));
        addOption(new CommandOption("proto", "protocol to use (amps, fix, nvfix, xml)\n\t\t   (type, prot are synonyms for backward compatibility)", "amps"));
        addOption(new CommandOption("authenticator", "Custom AMPS authenticator factory to use", false));
        addSynonym("type", "proto");
        addSynonym("prot", "proto");
    }

    abstract void execute(String argv[]);

    public String getName()
    {
        return this.name;
    }

    public void addOption(CommandOption command)
    {
        options.put(command.getName(), command);
        if(command.isRequired())
            required.add(command.getName());
        else
            optional.add(command.getName());
    }
    public void addSynonym(String synonym, String baseName)
    {
        synonyms.put(synonym, baseName);
    }
    public void addExample(String example)
    {
        examples.add(example);
    }

    public void addExtra(String extra)
    {
        extras.add(extra);
    }

    public void setUsage(String usage)
    {
        this.usage = usage;
    }

    protected void badUsage(String message)
    {
        help();
        System.err.println(message);
        System.exit(1);
    }

    void parseArgs(String argv[], int start)
    {
        String currentOption = null;
        for (int i = start; i < argv.length; ++i)
        {
            String current = argv[i];
            if(current.length() > 0 && current.charAt(0) == '-')
            {
                // We have a new option that begins with '-'.
                // First, let's make sure we handled the last one correctly...
                if(currentOption != null && !options.get(currentOption).hasValue())
                {
                    options.get(currentOption).setValue("true");
                }
                currentOption = current.substring(current.lastIndexOf('-') + 1);
                if(!options.containsKey(currentOption))
                {
                    if(synonyms.containsKey(currentOption))
                    {
                        currentOption = synonyms.get(currentOption);
                    }
                    else
                    {
                        badUsage("Invalid argument specified: '" + current + "'");
                    }
                }
            }
            else
            {
                if(currentOption == null)
                {
                    badUsage("Invalid argument specified: '" + current + "'");
                }
                options.get(currentOption).setValue(current);
            }
        }
        // Make sure we check the last option, in case there was no value
        if(currentOption != null && !options.get(currentOption).hasValue())
        {
            options.get(currentOption).setValue("true");
        }

        // Make sure all of the required ones are set...

        for (CommandOption option : options.values())
        {
            if(option.isRequired() && !option.hasValue())
            {
                // Oops, didn't set a required option, notify the user
                badUsage("The '" + option.getName() + "' option is required for the '" + getName()
                         + "' command.");
            }
        }
    }

    protected void dumpArgs()
    {
        for (CommandOption option : options.values())
        {
            System.out.print(option.getName() + " = ");
            if(option.hasValue())
            {
                System.out.println(option.getValue());
            }
            else
            {
                System.out.println("<unset>");
            }
        }
    }

    protected Authenticator createAuthenticator(String uri) throws AuthenticationException
    {
        CommandOption authenticatorOption = options.get("authenticator");
        String authFactoryName = "com.crankuptheamps.spark.DefaultAuthenticatorFactory";
        if (authenticatorOption.hasValue())
        {
            authFactoryName = authenticatorOption.getValue();
        }
        AuthenticatorFactory af = null;
        try
        {
            Class<?> c = Class.forName(authFactoryName);
            Constructor<?> ctor = c.getConstructor();

            af = (AuthenticatorFactory) ctor.newInstance();
        }
        catch (Exception e)
        {
            throw new com.crankuptheamps.client.exception.AuthenticationException("Could not find or create Authenticator Factory " + authFactoryName + ": " + e.toString(), e);
        }
        assert(af!=null);
        return af.createAuthenticator(uri);
    }

    private String fillString(String v, int s, char c)
    {
        StringBuffer buffer = new StringBuffer(s);
        buffer.append(v);
        while (buffer.length() < s)
            buffer.append(c);
        return buffer.toString();
    }

    public void help()
    {
        System.out.println("===============================");
        System.out.println("- Spark - AMPS client utility -");
        System.out.println("===============================");
        System.out.println("Usage:");
        System.out.println("");
        if(this.usage == null)
        {
            System.out.println("  spark " + getName() + " [options]");
        }
        else
        {
            System.out.println("  " + this.usage);
        }
        System.out.println("");

        if(required.size() > 0)
        {
            System.out.println("Required Parameters:");
            System.out.println("");
            for (String r : required)
            {
                System.out.println("  " + fillString(r, 10, ' ') + "-- "
                                   + options.get(r).getDescription());
            }
            System.out.println("");
        }
        if(optional.size() > 0)
        {
            System.out.println("Options:");
            System.out.println("");
            for (String r : optional)
            {
                System.out.print("  " + fillString(r, 14, ' ') + "-- "
                                   + options.get(r).getDescription());
                if(options.get(r).getDefaultValue()!=null)
                    System.out.println(" (default: "+options.get(r).getDefaultValue()+")");
                else
                    System.out.println();
            }
            System.out.println("");
        }

        for (String extra : extras)
        {
            System.out.println(extra);
        }

        if(examples.size() > 0)
        {
            if(examples.size() == 1)
                System.out.println("Example:");
            else
                System.out.println("Examples:");

            System.out.println("");

            for (String example : examples)
            {
                System.out.println(example);
            }
            System.out.println("");
        }
    }
}
