////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Scanner;
import java.util.UUID;

import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.Command;
import com.crankuptheamps.client.Message;
import com.crankuptheamps.client.MessageHandler;
import com.crankuptheamps.client.MessageStream;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.exception.ConnectionException;

public class SOWDeleteCommand extends SparkCommand
{
    String uriString                = null;
    String topic                    = null;
    String filter                   = null;
    private static final String LATIN1       = "ISO-8859-1";

    public SOWDeleteCommand(String name)
    {
        super(name);

        // These are the arguments this command recognizes
        addOption(new CommandOption("topic", "topic to delete from", true));
        addOption(new CommandOption("filter",  "content filter to use", false));
        addOption(new CommandOption("delimiter", "decimal value of message separator character (default 10)", false));
        addOption(new CommandOption("timeout", "timeout to wait for command to return in milliseconds", false));
        addOption(new CommandOption("file", "file to create delete records from, uses standard in when omitted and no filter is specified",
                false));
        
        // These are the examples...
        StringBuilder example = new StringBuilder(1024);
        example.append("  % ./spark sow_delete -server localhost:9003 -topic Trades -filter \"/55 = 'Transport.123'\"\n");
        example.append("    Connects to the AMPS instance listening on port 9003\n");
        example.append("    and deletes messages in topic 'Trades' that have\n");
        example.append("    tag 55 set to 'Transport.123'\n\n");
        example.append("  % ./spark sow_delete -server localhost:9003 -topic Trades -file deletes.fix\n");
        example.append("    Connects to the AMPS instance listening on port 9003\n");
        example.append("    and deletes messages in topic 'Trades' that match\n");
        example.append("    the messages found in the `deletes.fix' file.\n\n");
        example.append("  % ./spark sow_delete -server localhost:9003 -topic Trades\n");
        example.append("    Connects to the AMPS instance listening on port 9003\n");
        example.append("    and deletes messages in topic 'Trades' that match\n");
        example.append("    the messages read from standard input.\n");
        addExample(example.toString());
    }

    public void deleteByFilter(Client client, String topic, String filter, int timeout) throws AMPSException
    {
        long start = System.currentTimeMillis();
        Message m = client.execute(new Command("sow_delete").setTopic(topic).setFilter(filter).setTimeout(timeout)
                .addAckType(Message.AckType.Stats)).next();
        System.err.println("Deleted " + m.getMatches() + " records in " + (System.currentTimeMillis() - start) + " ms.");

    }

    public void deleteByData(Client client, String topic, String fileName, String delimiter, int timeout) throws AMPSException
    {
        long start = 0;
        Scanner scanner;
        String line = "";
        if(fileName != null)
        {
            //We're reading from a file
            try
            {
                scanner = new Scanner(new File(fileName), LATIN1);
            }
            catch (FileNotFoundException e)
            {
                badUsage("file '" + fileName + "' not found");
                return;
            }

        }
        else
        {
            // We're reading from standard in
            scanner = new Scanner(System.in);
        }

        start = System.currentTimeMillis();

        Command command = new Command(Message.Command.SOWDelete).setTopic(topic).addAckType(Message.AckType.Stats).setTimeout(timeout);

        if ( delimiter != null)
        {
            try
            {
                String strDelim = Character.toString((char)Integer.parseInt(delimiter));
                scanner.useDelimiter( strDelim );
            }
            catch (NumberFormatException nfex)
            {
                System.err.println("Delimiter is not a valid integer");
                System.exit(1);
            }
        }
        else
        {
            scanner.useDelimiter( System.getProperty("line.separator") );
        }

        long sent = 0, deleted = 0;
        while (scanner.hasNext())
        {
            line = scanner.next();
            command.setData(line);
            Message stats = client.execute(command).next();
            ++sent;
            deleted += stats.getRecordsDeleted();
        }
        System.err.format("Sent %d delete messages and deleted %d records in %d ms.\n",
                sent, deleted, System.currentTimeMillis() - start);
        // dispose all the resources after using them.
        scanner.close();
    }
    void execute(String argv[])
    {
        // Extract all of the command line arguments
        parseArgs(argv, 1);
        String[] userAndServer = options.get("server").getValue().split("@",2);
        String server = (userAndServer.length == 2) ? userAndServer[1] : userAndServer[0];
        topic          = options.get("topic").getValue();
        String type    = options.get("proto").getValue();
        String delimiter  = options.get("delimiter").getValue();
        String fileName = null;
        int timeout    = 5000;

        if(!type.equals("xml") && !type.equals("fix") && !type.equals("nvfix") && !type.equals("amps") && !type.equals("json"))
        {
            badUsage("prot argument needs to be 'fix', 'xml', 'nvfix', 'amps', or 'json'.");
        }
        if(options.get("file").hasValue())
        {
            fileName = options.get("file").getValue();
        }

        if(options.get("filter").hasValue())
        {
            filter = options.get("filter").getValue();
        }

        if(filter != null && fileName != null)
        {
            System.err.println("A filter and a filename cannot both be specified.");
            System.err.println("Specify a filter to delete a set of SOW records via a server-side query.");
            System.err.println("Specify a file name to delete a set of SOW records whose keys match those in the input file.");
            System.exit(1);
        }

        if(options.get("timeout").hasValue())
        {
            timeout = Integer.parseInt(options.get("timeout").getValue());
        }

        // Connect client to AMPS
        String userId = (userAndServer.length == 2) ? userAndServer[0] : System.getProperty("user.name");
        String clientName = "spark-sow-delete-" + userId + "-" + UUID.randomUUID().toString();
        Client client = new Client(clientName, Client.Version.AMPS_2);

        try
        {
            uriString = "tcp://" + userId + "@" + server + "/" + type;
            client.connect(uriString);
            client.logon(5000,createAuthenticator(uriString));
        }
        catch (ConnectionException e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }
        catch (Exception e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }

        try
        {
            if(filter == null)

            {
                deleteByData(client, topic, fileName, delimiter,timeout);
            } else
            {
                deleteByFilter(client,topic,filter,timeout);
            }
            client.close();
        } catch (AMPSException e)
        {
            System.err.format("Error: " + e.toString());
            System.exit(1);
        }
    }
}
