////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.Scanner;
import java.util.UUID;
import java.lang.Math;

import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.ClientDisconnectHandler;
import com.crankuptheamps.client.Message;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.exception.ConnectionException;

public class PublishCommand extends SparkCommand
{
    private int    messageCount = 0;
    private String uriString    = null;
    private static final String LATIN1       = "ISO-8859-1";

    public PublishCommand(String name)
    {
        super(name);

        // These are the arguments this command recognizes
        addOption(new CommandOption("topic", "topic to publish to", true));
        addOption(new CommandOption("file", "file to publish records from, standard in when omitted",
                                    false));
        addOption(new CommandOption("delta", "use delta publish", false));
        addOption(new CommandOption("delimiter", "decimal value of message separator character (default 10)", false));
        addOption(new CommandOption("rate", "decimal value used to send messages\n" +
                 "\t\tat a fixed rate.  '.25' implies 1 message every 4 seconds.\n" +
                 "\t\t'1000' implies 1000 messages per second.", false));
        // These are the examples...
        StringBuilder example = new StringBuilder(1024);
        example.append("  % ./spark publish -server localhost:9003 -topic Trades -file data.fix \n\n");
        example.append("    Connects to the AMPS instance listening on port 9003 and publishes records\n");
        example.append("    found in the 'data.fix' file to topic 'Trades'.");

        addExample(example.toString());
    }

    private class SparkDisconnectHandler implements ClientDisconnectHandler
    {
        public void invoke(Client client)
        {
            System.err.format("DISCONNECTED, trying to reconnect...%n");
            try
            {
                client.connect(uriString);
                client.logon(5000, createAuthenticator(uriString));
            }
            catch (AMPSException e)
            {
                System.err.println("Unable to re-connect to AMPS (" + e + ").");
                System.exit(1);
            }
        }
    }

    void execute(String argv[])
    {
        // Extract all of the command line arguments
        parseArgs(argv, 1);
        String[] userAndServer = options.get("server").getValue().split("@",2);
        String server = (userAndServer.length == 2) ? userAndServer[1] : userAndServer[0];
        String topic      = options.get("topic").getValue();
        String type       = options.get("proto").getValue();
        String delimiter  = options.get("delimiter").getValue();
        boolean delta     = options.get("delta").hasValue();
        String fileName   = null;
        long nano_interval   = 0;

        if(!type.equals("xml") && !type.equals("fix") && !type.equals("nvfix") && !type.equals("amps") && !type.equals("json"))
        {
            badUsage("prot argument needs to be 'fix', 'xml', 'nvfix', 'amps', or 'json'.");
        }

        if(options.get("file").hasValue())
        {
            fileName = options.get("file").getValue();
        }


        if (options.get("rate").hasValue())
        {
            nano_interval = Math.round( ( 1 / Double.valueOf(options.get("rate").getValue()) ) * Math.pow(10,9));
        }

        // Connect client to AMPS
        String userId = (userAndServer.length == 2) ? userAndServer[0] : System.getProperty("user.name");
        String clientName = "spark-publish-" + userId + "-" + UUID.randomUUID().toString();
        Client client = new Client(clientName, Client.Version.AMPS_2);
        client.setDisconnectHandler(new SparkDisconnectHandler());
        try
        {
            uriString = "tcp://" + userId + "@" + server + "/" + type;
            client.connect(uriString);
            client.logon(5000, createAuthenticator(uriString));
        }
        catch (ConnectionException e)
        {
            System.err.println("Unable to connect to AMPS (" + e + ").");
            System.exit(1);
        }

        long start = 0;
        try
        {
            Scanner scanner;
            String line = "";
            if(fileName != null)
            {
                //We're reading from a file
                scanner = new Scanner(new File(fileName), LATIN1);
            }
            else
            {
                // We're reading from standard in
                scanner = new Scanner(System.in);
            }

            start = System.currentTimeMillis();

            // Setup the Message object for publishing, reusing the
            // same message object for performance.
            try
            {
                Message message = client.allocateMessage();
                if(delta)
                {
                    message.setCommand(Message.Command.DeltaPublish);
                }
                else
                {
                    message.setCommand(Message.Command.Publish);
                }
                message.setTopic(topic);

                if ( delimiter != null)
                {
                    try
                    {
                        String strDelim = Character.toString((char)Integer.parseInt(delimiter));
                        scanner.useDelimiter( strDelim );
                    }
                    catch (NumberFormatException nfex)
                    {
                        System.err.println("Delimiter is not a valid integer");
                        System.exit(1);
                    }
                }
                else
                {
                    scanner.useDelimiter( System.getProperty("line.separator") );
                }

                long nextPubTime = System.nanoTime();
                while (scanner.hasNext())
                {

                    line = scanner.next();
                    message.setData(line);
                    while (nextPubTime > System.nanoTime()){ ;}// burn time to pace publishing
                    nextPubTime  += nano_interval;
                    client.send(message);
                    ++messageCount;
                }
            }
            catch (AMPSException e)
            {
                System.err.format("Error: disconnected (total messages published: %d) [%s]\n", messageCount,e);
                System.exit(1);
            }

            // dispose all the resources after using them.
            scanner.close();

        }
        catch (FileNotFoundException e)
        {
            badUsage("file '" + fileName + "' not found");
        }

        client.close();

        long elapsed = System.currentTimeMillis() - start;
        System.err.format("total messages published: %d (%.2f/s)\n", messageCount, 1000.0
                          * messageCount / elapsed);
    }
}
