package com.crankuptheamps.client;

import java.net.Socket;
import java.net.URI;
import java.nio.ByteBuffer;
import java.beans.ExceptionListener;
import java.util.Properties;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.crankuptheamps.client.Message.SerializationResult;
import com.crankuptheamps.client.exception.AlreadyConnectedException;
import com.crankuptheamps.client.exception.ConnectionRefusedException;
import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.InvalidURIException;
import com.crankuptheamps.client.exception.RetryOperationException;

public class TCPTransport implements Transport
{
    protected TCPTransportImpl _impl         = null;
    private Protocol           _protocol  = null;
    private final Lock         _sendLock     = new ReentrantLock();
    private ByteBuffer         _sendBuffer   = ByteBuffer.allocate(4096);

    // Other clients use daemon threads by default, but
    // we have customer that depends on non-daemon threads (i.e. they
    // want the program to continue running even once main() exits.)
    private static boolean _useDaemonThreads = false;

    protected TCPTransportImpl constructTransportImpl(Protocol protocol, Properties properties)
    {
        return new TCPTransportImpl(protocol, properties, new DefaultTransportFilter());
    }
    public TCPTransport(Protocol protocol, Properties properties)
    {
        this._protocol = protocol;
        this._impl = constructTransportImpl(protocol,properties);
    }

    public TCPTransport(Protocol msgType)
    {
        this(msgType, null);
    }

    public static void setDaemon(boolean daemonThreads)
    {
        TCPTransport._useDaemonThreads = daemonThreads;
    }

    public static boolean isDaemon()
    {
        return TCPTransport._useDaemonThreads;
    }

    public static TCPTransport createTransport(Protocol messageType)
    {
        return new TCPTransport(messageType);
    }

    public void setMessageHandler(MessageHandler ml)
    {
        this._impl.setMessageHandler(ml);
    }

    public void setDisconnectHandler(TransportDisconnectHandler dh)
    {
        this._impl.setDisconnectHandler(dh);
    }

    public void setExceptionListener(ExceptionListener exceptionListener)
    {
        this._impl.setExceptionListener(exceptionListener);
    }

    public void setTransportFilter(TransportFilter filter)
    {
        _impl.setTransportFilter(filter);
    }
    public void connect(URI uri) throws ConnectionRefusedException, AlreadyConnectedException, InvalidURIException
    {
        this._impl.connect(uri);
    }

    public void close()
    {
        this._impl.disconnect();
    }

    public void disconnect()
    {
        this._impl.disconnect();
    }

    public void handleCloseEvent(int failedVersion_, String message, Exception e) throws DisconnectedException, RetryOperationException
    {
        this._impl.handleCloseEvent(failedVersion_, message, e);
    }

    public void sendWithoutRetry(Message message) throws DisconnectedException
    {
        this._sendLock.lock();
        try
        {
            _sendBuffer.clear();
            // Reserve space for a 4-byte int at the
            // beginning for the size of the message
            _sendBuffer.position(4);
            SerializationResult sr = message.serialize(_sendBuffer);
            if(sr == Message.SerializationResult.BufferTooSmall)
            {
                // Buffer was too small, let's resize and retry
                _sendBuffer = ByteBuffer.allocate(2 * _sendBuffer.capacity());
                sendWithoutRetry(message);
                return;
            }
            else if(sr == SerializationResult.OK)
            {
                // Write size integer now that we know the size
                _sendBuffer.putInt(0, _sendBuffer.position() - 4);
                // Flip buffer for reading
                _sendBuffer.flip();
                // Write buffer to socket
                this._impl.send(_sendBuffer);
            }
        }
        finally
        {
            this._sendLock.unlock();
        }
    }

    public void send(Message message) throws DisconnectedException
    {
        this._sendLock.lock();
        try
        {
            int currentVersion = getVersion();
            try
            {
                sendWithoutRetry(message);
            }
            catch(DisconnectedException ex)
            {
                try
                {
                    this.handleCloseEvent(currentVersion, "Exception occured while sending", ex);
                }
                catch(RetryOperationException r)
                {
                    ; // retry
                }
                throw ex;
            }

        }
        finally
        {
            this._sendLock.unlock();
        }
    }

    public Message allocateMessage()
    {
        return this._protocol.allocateMessage();
    }

    public long writeQueueSize() throws DisconnectedException
    {
        try
        {
            return this._impl.writeQueueSize();
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
    }

    public long readQueueSize() throws DisconnectedException
    {
        try
        {
            return this._impl.writeQueueSize();
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
    }

    public long flush() throws DisconnectedException
    {
        try
        {
            return this._impl.flush();
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
    }

    public long flush(long timeout) throws DisconnectedException
    {
        try
        {
            return this._impl.flush(timeout);
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
    }

    public Socket socket()
    {
        try
        {
            return this._impl.socket();
        }
        catch(Exception e)
        {
            return null;
        }
    }

    public int getVersion()
    {
        return this._impl._connectionVersion;
    }

    public void setReadTimeout(int readTimeout_)
    {
        this._impl.setReadTimeout(readTimeout_);
    }
}
