///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.io.IOException;
import java.lang.Long;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.*;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;
import com.crankuptheamps.client.exception.*;


/** Implements a bookmark store useful for handling
 * server failover scenarios, but without a backing store to recover from subscriber failure.
 */
public class MemoryBookmarkStore implements BookmarkStore
{
    // We use an in-memory array to remember the order in which bookmarks
    // arrive on a given subscription.  When the user has discard()ed all of the
    // bookmarks up to a certain bookmark B, we log B, and return it the next time
    // someone calls getMostRecent().
    static class Subscription implements com.crankuptheamps.client.Subscription
    {
        // This subscription's ID.
        Field _sub;

        // The last persisted bookmark
        private BookmarkField _lastPersisted;

        BookmarkRingBuffer _ring = new BookmarkRingBuffer();

        // A set of all of the undiscarded entries recovered that
        // had been received after the one returned by getMostRecent().
        // When these come in again, we won't log them a 2nd time.
        HashMap<Field, Long> _recovered = new HashMap<Field, Long>();

        // The per-subscription memory of what we've seen from publishers
        HashMap<Long, Long> _publishers = new HashMap<Long, Long>();

        // A map of active bookmark indices to their new location after a reconnect
        HashMap<Long, Long> _relocatedActiveBookmarks = new HashMap<Long, Long>();

        public Subscription(Field subscriptionId)
        {
            _sub = subscriptionId.copy();
            _ring.setSubId(_sub);
        }
        public Subscription()
        {
        }
        public void setSubscription(Field subId)
        {
            _sub = subId.copy();
            _ring.setSubId(_sub);
        }

        public synchronized long log(BookmarkField bookmark)
        {
            // Check to see if this is a recovered bookmark.
            // If so, just return that one -- don't re-log.
            final Long recoveredValue = _recovered.remove(bookmark);
            if (recoveredValue != null) {
                long index = _ring.relog(recoveredValue, bookmark);
                _relocatedActiveBookmarks.put(recoveredValue, index);
                return index;
            }

            return _ring.log(bookmark);
        }

        public synchronized void discard(long index)
        {
            long idx = index;
            Long newIndex = _relocatedActiveBookmarks.remove(index);
            // In case something's been recovered more than once, get full indirection
            while (newIndex != null)
            {
                _ring.discard(idx);
                idx = newIndex;
                newIndex = _relocatedActiveBookmarks.remove(newIndex);
            }
            _ring.discard(idx);
        }

        // Check to see if this message is older than the most recent one seen,
        // and if it is, check if it discarded.
        public synchronized boolean isDiscarded(BookmarkField bookmark)
        {
            Long recoveredIndex = _recovered.get(bookmark);
            if (recoveredIndex != null)
            {
                BookmarkRingBuffer.Entry entry = _ring.getByIndex(recoveredIndex);
                boolean active = (entry == null) ? false : entry.isActive();
                if (active)
                {
                    return false;
                }
                if ((entry == null || (!entry.isActive() && entry.isPersisted())) &&
                         _ring.getStartIndex() == BookmarkRingBuffer.UNSET_INDEX)
                {
                    // This will update most recent if necessary
                    _ring.discard(recoveredIndex);
                }
                else
                {
                    long newIndex = _ring.relog(recoveredIndex, bookmark);
                    if (active)
                    {
                        _relocatedActiveBookmarks.put(recoveredIndex, newIndex);
                    }
                }
                _recovered.remove(bookmark);
                return true;
            }
            long publisher = bookmark.getPublisherId();
            long sequence = bookmark.getSequenceNumber();
            if(!_publishers.containsKey(publisher) || _publishers.get(publisher) < sequence)
            {
                _publishers.put(publisher, sequence);
                return false;
            }
            BookmarkRingBuffer.Entry entry =
                    _ring.find(bookmark);
            if(entry != null)
            {
                return !entry.isActive();
            }

            return true; // message is totally discarded
        }

        public synchronized Field getMostRecent()
        {
            // _recent is the most recent bookmark.
            // when this is called, we'll take a moment to update the list of things recovered,
            // so we don't accidentally log anything we ought not to.
            updateRecovery();
            return _ring.getLastDiscarded();
        }

        public synchronized Field getMostRecentList()
        {
            // when this is called, we'll take a moment to update the list of things
            // recovered, so we don't accidentally log anything we ought not to.
            updateRecovery();
            BookmarkField lastDiscarded = (BookmarkField)_ring.getLastDiscarded();
            boolean useLastDiscarded = (lastDiscarded != null && !lastDiscarded.isNull());
            long lastDiscardedPub = 0;
            long lastDiscardedSeq = 0;
            boolean useLastPersisted = (_lastPersisted != null && !_lastPersisted.isNull());
            long lastPersistedPub = 0;
            long lastPersistedSeq = 0;
            if (useLastPersisted)
            {
                lastPersistedPub = ((BookmarkField)_lastPersisted).getPublisherId();
                lastPersistedSeq = ((BookmarkField)_lastPersisted).getSequenceNumber();
            }
            if (useLastDiscarded)
            {
                if (_ring.isEmpty() && useLastPersisted)
                {
                    useLastDiscarded = false;
                }
                else
                {
                    lastDiscardedPub = lastDiscarded.getPublisherId();
                    lastDiscardedSeq = lastDiscarded.getSequenceNumber();
                    // Only use one if they are the same publisher
                    if (useLastPersisted && (lastPersistedPub == lastDiscardedPub) &&
                        (lastPersistedSeq < lastDiscardedSeq))
                    {
                        useLastPersisted = (lastPersistedSeq < lastDiscardedSeq);
                        useLastDiscarded = !useLastPersisted;
                    }
                }
            }
            StringBuilder recentStr = new StringBuilder();
            if (useLastDiscarded)
            {
                recentStr.append((lastDiscarded).getValue(Charset.forName("ISO-8859-1").newDecoder()));
            }
            if (!useLastPersisted && !useLastDiscarded)
            {
                if (_publishers.isEmpty())
                {
                    // Set last persisted to EPOCH and return it
                    _lastPersisted.setValue(Client.Bookmarks.EPOCH, Charset.forName("ISO-8859-1").newEncoder());
                    return _lastPersisted;
                }
                Iterator it = _publishers.entrySet().iterator();
                while (it.hasNext())
                {
                    if (recentStr.length() > 0) recentStr.append(",");
                    Map.Entry pairs = (Map.Entry)it.next();
                    long pubId = (Long)pairs.getKey();
                    long seq = (Long)pairs.getValue();
                    recentStr.append(pubId).append("|");
                    recentStr.append(seq).append("|");
                }
            }
            if (useLastPersisted)
            {
                if (recentStr.length() > 0) recentStr.append(",");
                recentStr.append(((BookmarkField)_lastPersisted).getValue(Charset.forName("ISO-8859-1").newDecoder()));
            }
            BookmarkField recentList = new BookmarkField();
            recentList.setValue(recentStr.toString(), Charset.forName("ISO-8859-1").newEncoder());
            return recentList;
        }

        private void updateRecovery()
        {
            long end = _ring.getEndIndex();
            for (long index = _ring.setRecovery(); index < end; ++index)
            {
                BookmarkRingBuffer.Entry entry = _ring.getByIndex(index);
                if(entry.isActive() || !entry.isPersisted())
                {
                    _recovered.put(entry.getBookmark(), index);
                }
            }
        }

        public synchronized void persisted(long bookmark)
        {
            _ring.persisted(bookmark);
        }

        public synchronized void persisted(BookmarkField bookmark)
        {
            _ring.persisted(bookmark);
        }

        public synchronized void setLastPersisted(long bookmark)
        {
            BookmarkRingBuffer.Entry entry = _ring.getByIndex(bookmark);
            if (entry == null)
                return;
            if (_lastPersisted != null) _lastPersisted.reset();
            _lastPersisted = entry.getBookmark().copy();
        }

        public synchronized void setLastPersisted(BookmarkField bookmark)
        {
            if (!_ring.persistedAcks() && _lastPersisted != null &&
                bookmark.getPublisherId() == _lastPersisted.getPublisherId() &&
                bookmark.getSequenceNumber() <= _lastPersisted.getSequenceNumber())
            {
                return;
            }
            if (_lastPersisted != null) _lastPersisted.reset();
            _lastPersisted = bookmark.copy();
        }

        public synchronized void noPersistedAcks()
        {
            _ring.noPersistedAcks();
        }

        public synchronized void setPersistedAcks()
        {
            _ring.setPersistedAcks();
        }

        public synchronized long getOldestBookmarkSeq()
        {
            return _ring.getStartIndex();
        }

        public void setResizeHandler(BookmarkStoreResizeHandler handler, BookmarkStore store)
        {
            _ring.setResizeHandler(handler, store);
        }
    }

    HashMap<Field, Subscription> _subs = new HashMap<Field, Subscription>();
    BookmarkStoreResizeHandler _resizeHandler = null;
    private int _serverVersion = Client.MIN_MULTI_BOOKMARK_VERSION;

    // Subscriptions take a while to initialize, so we avoid doing this in the message handler.
    // number of subscriptions we pre-initialize.
    Pool<Subscription> _pool;
    public MemoryBookmarkStore()
    {
        _pool = new Pool<Subscription>(Subscription.class, 1);
    }

    /**
     * Initialize self with a target number of subscriptions to store
     * @param targetNumberOfSubscriptions the number of subscriptions this store will need to track
     */
    public MemoryBookmarkStore(int targetNumberOfSubscriptions)
    {
        _pool = new Pool<Subscription>(Subscription.class, targetNumberOfSubscriptions);
    }
    public long log(Message message) throws AMPSException
    {
        BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
        Subscription sub = (MemoryBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
        long seqNo = sub.log(bookmark);
        message.setBookmarkSeqNo(seqNo);
        return seqNo;
    }

    public void discard(Field subId, long bookmarkSeqNo) throws AMPSException
    {
        find(subId).discard(bookmarkSeqNo);
    }

    public void discard(Message message) throws AMPSException
    {
        long bookmark = message.getBookmarkSeqNo();
        Subscription sub = (MemoryBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
    	sub.discard(bookmark);
    }

    public Field getMostRecent(Field subId) throws AMPSException
    {
        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
            return find(subId).getMostRecentList();
        else
            return find(subId).getMostRecent();
    }

    public boolean isDiscarded(Message message) throws AMPSException
    {
        BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
        Subscription sub = (MemoryBookmarkStore.Subscription)message.getSubscription();
        if (sub == null)
        {
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();
            sub = find(subId);
            message.setSubscription(sub);
        }
        return sub.isDiscarded(bookmark);
    }

    public void persisted(Field subId, long bookmark) throws AMPSException
    {
        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
            find(subId).setLastPersisted(bookmark);
        else
            find(subId).persisted(bookmark);
    }

    public void persisted(Field subId, BookmarkField bookmark) throws AMPSException
    {
        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
            find(subId).setLastPersisted(bookmark);
        else
            find(subId).persisted(bookmark);
    }

    public void noPersistedAcks(Field subId) throws AMPSException
    {
        find(subId).noPersistedAcks();
    }

    private synchronized Subscription find(Field subId)
    {
        Subscription s = _subs.get(subId);
        if(s==null)
        {
            s = _pool.get();
            s.setSubscription(subId);
            s.setResizeHandler(_resizeHandler, this);
            if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION ||
                _serverVersion < Client.MIN_PERSISTED_BOOKMARK_VERSION)
                s.noPersistedAcks();
            _subs.put(subId.copy(), s);
        }
        return s;
    }

    public synchronized void purge() throws AMPSException
    {
        _subs = new HashMap<Field, Subscription>();
    }

    public synchronized long getOldestBookmarkSeq(Field subId) throws AMPSException
    {
        long retVal = 0;
        retVal = find(subId).getOldestBookmarkSeq();
        return retVal;
    }

    public void setResizeHandler(BookmarkStoreResizeHandler handler)
    {
        _resizeHandler = handler;
        Iterator it = _subs.entrySet().iterator();
        while (it.hasNext())
        {
            Map.Entry pairs = (Map.Entry)it.next();
            ((Subscription)pairs.getValue()).setResizeHandler(handler, this);
        }
    }

    public void setServerVersion(int version)
    {
        if (_serverVersion == version) return;
        _serverVersion = version;
        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION ||
            _serverVersion < Client.MIN_PERSISTED_BOOKMARK_VERSION)
        {
            Iterator it = _subs.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry pairs = (Map.Entry)it.next();
                ((Subscription)pairs.getValue()).noPersistedAcks();
            }
        }
        else
        {
            Iterator it = _subs.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry pairs = (Map.Entry)it.next();
                ((Subscription)pairs.getValue()).setPersistedAcks();
            }
        }
    }
}
