////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////
package com.crankuptheamps.client;

import java.io.IOException;

import com.crankuptheamps.client.exception.*;

/**
 *
 * A highly-available AMPS Client object that automatically reconnects and
 * re-subscribes to AMPS instances upon disconnect.
 *
 */
public class HAClient extends Client
{
    private volatile boolean _disconnected = false;
    private final static int DefaultTimeout = 10000;
    private final static int DefaultReconnectInterval = 200;

    private int _timeout = DefaultTimeout;
    private int _reconnectDelay = DefaultReconnectInterval;
    private int _heartbeatInterval = 0; // off
    private int _readTimeout = 0;
    private volatile boolean _firstConnect = true;

    /**
     * Returns the time delay between reconnect events.
     * @return The time waited between reconnect events, in milliseconds.
     */
    public int getReconnectDelay()
    {
        return _reconnectDelay;
    }

    /**
     * Sets the time delay between reconnect events.
     * @param reconnectInterval The time waited between reconnect events, in milliseconds.
     */
    public void setReconnectDelay(int reconnectInterval)
    {
        this._reconnectDelay = reconnectInterval;
    }

    /**
     * Sets the timeout used for logging on and resubscribing, once a connection
     * is re-established.
     *
     * @param timeout_
     *            The timeout value, in milliseconds.
     */
    public void setTimeout(int timeout_)
    {
        _timeout = timeout_;
    }

    /**
     * Returns the timeout used for logging on and resubscribing, once a
     * connection is re-established.
     *
     * @return The timeout value in millseconds.
     */
    public int getTimeout()
    {
        return _timeout;
    }

    // Overridden to store the value used, so we can apply it to future connections.
    @Override
    public void setHeartbeat(int intervalSeconds_, int timeoutSeconds_) throws DisconnectedException
    {
        _heartbeatInterval = intervalSeconds_;
        _readTimeout = timeoutSeconds_;
        super.setHeartbeat(intervalSeconds_, timeoutSeconds_);
    }

    public static class HADisconnectHandler implements ClientDisconnectHandler, ClientDisconnectHandler2
    {

        public void invoke(Client client)
        {
            // This is only here to support subclasses of HADisconnectHandler, prior
            // to the addition of the additional argument to invoke(), below.
        }

        public void invoke(Client client_, Exception e_)
        {
            // a no-op, unless you've derived HADisconnectHandler.
            invoke(client_);

            HAClient haClient = (HAClient) client_;
            haClient.getServerChooser().reportFailure(
                    e_, client_.getConnectionInfo());

            try
            {
                haClient.connectAndLogonInternal();

            } catch (Exception e)
            {
                if (client_.exceptionListener != null)
                {
                    client_.exceptionListener.exceptionThrown(e);
                }
                return;
            }
        }

        public HADisconnectHandler()
        {
        }

    }

    /**
     * Returns the current {@link ServerChooser}.
     *
     * @return The current ServerChooser.
     */
    public ServerChooser getServerChooser()
    {
        return _serverChooser;
    }

    /**
     * Sets a {@link ServerChooser} for self.
     *
     * @param serverChooser_
     *            The ServerChooser that will be used by self to reconnect.
     */
    public void setServerChooser(ServerChooser serverChooser_)
    {
        this._serverChooser = serverChooser_;
    }

    ServerChooser _serverChooser;
    HADisconnectHandler _disconnectHandler;

    /**
     * Constructs a new HAClient.
     *
     * @param name
     *            The client name, passed to the server to uniquely identify
     *            this client across sessions.
     */
    public HAClient(String name)
    {
        super(name);
        super.setDisconnectHandler(new HADisconnectHandler());
        super.setSubscriptionManager(new MemorySubscriptionManager());
    }

    /**
     * Creates a new memory-backed highly-available client. This client will
     * automatically reconnect and re-subscribe, and uses memory to ensure
     * messages are not lost and duplicates are processed appropriately.
     *
     * @param name
     *            The client name, passed to the server to uniquely identify
     *            this client across sessions.
     * @return A new memory-backed HAClient instance.
     */
    public static HAClient createMemoryBacked(String name) throws StoreException
    {
        HAClient client = new HAClient(name);
        client.setBookmarkStore(new MemoryBookmarkStore());
        client.setPublishStore(new MemoryPublishStore(10000));

        return client;
    }

    /**
     * Creates a new highly available client backed by disk. This client will
     * automatically reconnect and re-subscribe, and uses disk to ensure
     * messages are not lost and duplicates are processed appropriately.
     *
     * @param name
     *            The client name, passed to the server to uniquely identify
     *            this client across sessions.
     * @param publishLog
     *            The path to a file used to store outgoing messages. This file
     *            will be created if it does not exist.
     * @param initialPublishCapacity
     *            The initial size (in 2k blocks) of the publish store.  Choosing
     *            an optimal value of initialPublishCapacity can prevent costly
     *            resizes, when servers or networks slow down.
     * @param subscribeLog
     *            The path to a file used to log incoming message bookmarks.
     *            This file will be created if it does not exist.

     *
     * @return A new HAClient instance.
     */
    public static HAClient createFileBacked(String name, String publishLog,
            int initialPublishCapacity, String subscribeLog) throws IOException, StoreException
    {
        HAClient client = new HAClient(name);
        client.setBookmarkStore(new LoggedBookmarkStore(subscribeLog));
        client.setPublishStore(new PublishStore(publishLog, initialPublishCapacity));
        return client;
    }

    /**
     * Creates a new highly available client backed by disk. This client will
     * automatically reconnect and re-subscribe, and uses disk to ensure
     * messages are not lost and duplicates are processed appropriately.
     *
     * @param name
     *            The client name, passed to the server to uniquely identify
     *            this client across sessions.
     * @param publishLog
     *            The path to a file used to store outgoing messages. This file
     *            will be created if it does not exist.
     * @param subscribeLog
     *            The path to a file used to log incoming message bookmarks.
     *            This file will be created if it does not exist.
     * @return A new HAClient instance.
     */
    public static HAClient createFileBacked(String name, String publishLog,
            String subscribeLog) throws IOException, StoreException
    {
        HAClient client = new HAClient(name);
        client.setBookmarkStore(new LoggedBookmarkStore(subscribeLog));
        client.setPublishStore(new PublishStore(publishLog));
        return client;
    }

    /**
     * Connects to the next server chosen for us by our {@link ServerChooser}.
     * Will continue attempting to connect and logon to each successive
     * server returned by the ServerChooser until the connection suceeds
     * or the ServerChooser returns an empty URI.
     *
     * @throws ConnectionException
     *             An exception indicating no AMPS instance could be connected
     *             to.
     */
    public void connectAndLogon() throws ConnectionException
    {
        _disconnected = false;
        while(true)
        {
            connectAndLogonInternal();

            // When manually called, we must re-subscribe manually as well.
            try
            {
                this.getSubscriptionManager().resubscribe(this);
                return;
            } catch (AMPSException subException)
            {
                super.disconnect();
                _serverChooser.reportFailure(subException, getConnectionInfo());
            }
        }

    }

    private void connectAndLogonInternal() throws ConnectionException
    {
        // get the next pair from the list
        if (_serverChooser == null)
        {
            throw new ConnectionException(
                    "No server chooser registered with this HAClient.");
        }
        // block any extra disconnect handling while we are in this loop.
        ClientDisconnectHandler currentHandler = super.getDisconnectHandler();
        super.setDisconnectHandler(new DefaultDisconnectHandler());
        try
        {
            while (!_disconnected)
            {
                String uri = _serverChooser.getCurrentURI();

                if (uri == null)
                {
                    throw new ConnectionException(
                            "No AMPS instances available for connection. " + _serverChooser.getError());
                }
                try
                {
                    if (this._firstConnect == false)
                        Thread.sleep(this._reconnectDelay);
                    else
                        this._firstConnect = false;
                    Authenticator authenticator = _serverChooser
                            .getCurrentAuthenticator();
                    super.connect(uri);
                    super.getTransport().getVersion();
                    super.logon(_timeout, authenticator);
                    super.setHeartbeat(_heartbeatInterval, _readTimeout);
                    _serverChooser.reportSuccess(getConnectionInfo());
                    return;
                } catch (Exception exception)
                {
                    // We need to substitute the "current" URI with the one
                    // returned by the serverChooser; the URI may not yet be set on the
                    // client.
                    ConnectionInfo ci = getConnectionInfo();
                    ci.put("client.uri", uri);
                    _serverChooser.reportFailure(exception, ci);
                    // Try disconnecting -- we might be partially connected?
                    try
                    {
                        super.disconnect();
                    } catch (Exception e)
                    {
                        if (exceptionListener != null)
                        {
                            exceptionListener.exceptionThrown(e);
                        }
                    }
                }
            }
        }
        finally
        {
          super.setDisconnectHandler(currentHandler);
        }
    }
    @Override
    public void connect(String uri) throws ConnectionException
    {
        _disconnected = false;
        connectAndLogon();
    }

    @Override
    public void disconnect()
    {
        _disconnected = true;
        super.disconnect();
    }

    /**
     * Assembles a new ConnectionInfo with the state of this client and
     * associated classes.
     * @return A new ConnectionInfo object.
     */
    @Override
    public ConnectionInfo getConnectionInfo()
    {
        ConnectionInfo ci = super.getConnectionInfo();
        ci.put("haClient.reconnectDelay", this._reconnectDelay);
        ci.put("haClient.timeout", this._timeout);
        return ci;
    }
}
