////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import com.crankuptheamps.client.exception.CommandException;

public class FIXBuilder
{
    private byte[] buffer   = null;
    private int    size     = 0;
    private int    capacity = 1024;

    private byte   fieldSeparator;

    public FIXBuilder(int capacity, byte fieldSeparator)
    {
        this.fieldSeparator = fieldSeparator;
        this.capacity = capacity;
        this.buffer = new byte[capacity];
    }

    
    /**
     *  Clears self.
     */
    public void clear()
    {
        size = 0;
    }
    
    /**
     * Returns the number of bytes in the byte array containing the FIX message.
     *
     * @return number of valid bytes in the byte array
     */
    public int getSize()
    {
        return this.size;
    }

    /**
     * Returns the byte array containing the FIX message. The number of valid
     * bytes within the buffer is returned from getSize().
     *
     * @return byte array containing the FIX message
     */
    public byte[] getBytes()
    {
        return buffer;
    }

    private int ruLog10(int tag)
    {
        long scalar = 10;
        for (int j = 1; j < 20; ++j)
        {
            if(tag < scalar)
            {
                return j;
            }
            scalar = scalar * 10;
        }
        return 0;
    }

    private void checkCapacity(int bytesNeeded)
    {
        if(capacity - size < bytesNeeded)
        {
            // Not enough capacity left, must resize
            while (capacity - size < bytesNeeded)
            {
                capacity = capacity * 2;
            }

            byte[] newbuf = new byte[capacity];
            System.arraycopy(buffer, 0, newbuf, 0, size);

            buffer = newbuf;
        }
    }

    /**
     * Appends a tag and and value pair contained within a byte buffer to the FIX message.
     *
     * @param tag
     *           the integer FIX tag to append
     * @param value
     *           the byte buffer containing the FIX value to append.
     * @param offset
     *           the starting location of the value inside the byte buffer.
     * @param length
     *           the length of the value inside the byte buffer.
     * @throws IllegalArgumentException
     *           if the tag argument is negative.
     * @return
     *           a reference to this object.
     */
    public FIXBuilder append(int tag, byte[] value, int offset, int length)
    throws CommandException
    {
        if(tag < 0)
            throw new IllegalArgumentException("negative tag used in FIXBuilder");

        int tagSize = ruLog10(tag);
        int sizeNeeded = tagSize + length + 2;
        checkCapacity(sizeNeeded);

        // Tag
        int writeIndex = size + tagSize;
        for (int j = 1; j <= tagSize; ++j)
        {
            buffer[writeIndex - j] = (byte) (48 + (tag % 10));
            tag = tag / 10;
        }

        // Equal sign
        buffer[writeIndex] = '=';
        ++writeIndex;

        // value
        System.arraycopy(value, offset, buffer, writeIndex, length);
        writeIndex += length;

        buffer[writeIndex] = fieldSeparator;
        size += sizeNeeded;
        return this;
    }

    /**
     * Appends a tag and value pair to the FIX message.
     *
     * @param tag
     *           the integer FIX tag to append
     * @param value
     *           the FIX value for the tag to append. The value will be converted
     *           to an ISO-8859-1 byte array for writing.
     * @throws IllegalArgumentException
     *            if the tag argument is negative.
     * @throws CommandException
     *            if the tag or value argument is not convertible to ISO-8859-1
     * @return a reference to this object.
     */
    public FIXBuilder append(int tag, String value) throws CommandException
    {
        if(tag < 0)
            throw new IllegalArgumentException("negative tag used in FIXBuilder");

        int tagSize = ruLog10(tag);
        byte[] valueBytes;
        try
        {
            valueBytes = value.getBytes("ISO-8859-1");
        }
        catch (java.io.UnsupportedEncodingException ioex)
        {
            throw new CommandException("Unable to encode value '" + value
                                       + "' to byte array in FIXBuilder.\n" + ioex.toString());
        }
        int sizeNeeded = tagSize + valueBytes.length + 2;
        checkCapacity(sizeNeeded);

        // Tag
        int writeIndex = size + tagSize;
        for (int j = 1; j <= tagSize; ++j)
        {
            buffer[writeIndex - j] = (byte) (48 + (tag % 10));
            tag = tag / 10;
        }

        // Equal sign
        buffer[writeIndex] = '=';
        ++writeIndex;

        // Value
        System.arraycopy(valueBytes, 0, buffer, writeIndex, valueBytes.length);
        writeIndex += valueBytes.length;

        buffer[writeIndex] = fieldSeparator;
        size += sizeNeeded;

        return this;
    }

}
