package com.crankuptheamps.client;

import com.crankuptheamps.client.exception.TimedOutException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.TreeMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.TimeUnit;
import java.util.zip.CRC32;

import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.StoreException;

public class BlockPublishStore implements Store
{
    private int _blocksPerRealloc = 10000;
    public static class ByteSequence
    {
        public byte[] array;
        public long offset, len;
        public ByteSequence()
        {
            this.array = null;
            this.offset = this.len = 0;
        }
        public ByteSequence(byte[] array, long offset, long len)
        {
            this.array = array;
            this.offset = offset;
            this.len = len;
        }
    }
    
    public interface Buffer
    {
        public long getSize() throws IOException;
        public void setSize(long newSize) throws IOException;
        public long getPosition() throws IOException;
        public void setPosition(long position) throws IOException;
        
        public void putByte(byte b) throws IOException;
        public byte getByte() throws IOException;
        public void putInt(int i) throws IOException;
        public int getInt() throws IOException;
        public void putLong(long l) throws IOException;
        public long getLong() throws IOException;
        
        public void putBytes(ByteSequence bytes) throws IOException;
        public void getBytes(ByteSequence outBytes) throws IOException;
        
        public void zero(long offset, int length) throws IOException;
        public void setResizeHandler(PublishStoreResizeHandler handler, Store store);
    }
    
    static class Block
    {
        public int index;
        public Block nextInChain;
        public Block nextInList;
        public long sequenceNumber;
        public static final int SIZE = 2048;  // finely honed.
        public static final int BLOCK_HEADER_SIZE = 32;
        public static final int BLOCK_DATA_SIZE = Block.SIZE - BLOCK_HEADER_SIZE;
        public static final int CHAIN_HEADER = 5;
        public Block(int index) { this.index = index; }
        public long getPosition() { return SIZE * index; }
        public String toString()
        {
            return String.format("block %d sequenceNumber %d", index, sequenceNumber);
        }
    }
    ReadWriteLock _bufferLock;
    
    // A buffer acts as a random access file -- mmap, memory only, and file implementations exist.
    protected Buffer _buffer;
    
    // This system uses intrusive lists.  _freeList is the list of free blocks in the file.
    // _usedList is the list of the used head blocks.  Used non-head blocks are only kept
    // alive by "chain" references from the head block.
    // Used list is ordered where the start is the LEAST recently used.
    Block _freeList, _usedList, _endOfUsedList;
    
    // Save this so we're not GCing stuff.
    CRC32 _crc = null;
    
    // We use this _argument thing since Java doesn't have output parameters, but we sort of need them.
    ByteSequence _argument = new ByteSequence();

    // We use this for pulling a command id from the buffer when reading sow_delete
    ByteSequence _commandId = new ByteSequence();
    
    // We use this for pulling a correlation id from the buffer
    ByteSequence _correlationId = new ByteSequence();
    
    // When a message spans multiple blocks we need to reassemble it
    // into one place before replaying.  We use this buffer to do so,
    // allocating on demand.
    byte[] _readBuffer;
    
    // Count of used messages -- entries on the used list.
    volatile int _usedCount;
    
    // Write assembly area
    ArrayStoreBuffer _sb = null;
    
    protected static final int HAS_EXTENDED_METADATA = 0x04;
    
    
    protected BlockPublishStore(Buffer buffer, int blocksPerRealloc, boolean isAFile)
    {
        this._blocksPerRealloc = blocksPerRealloc;
        this._buffer = buffer;
        if(isAFile)
        {
            this._crc = new CRC32();
            this._sb = new ArrayStoreBuffer();
            try
            {
                this._sb.setSize(Block.SIZE);
            } catch (Exception e)
            {
                this._sb = null;
            }           
        }
    }
    protected BlockPublishStore(Buffer buffer, int blocksPerRealloc)
    {
        this(buffer, blocksPerRealloc, true);
        
    }
    protected BlockPublishStore(Buffer buffer)
    {
        this(buffer, 10000);
    }

    
    public void store(long index, int operation, byte[] topic,
                                   long topicOffset, long topicLength,
                                   byte[] data, long dataOffset, long dataLength,
                                   byte[] correlationId, long correlationIdOffset, long correlationIdLength)
                                       throws StoreException
    {
        _store(index, operation, topic, topicOffset, topicLength,
               data, dataOffset, dataLength,
               correlationId, correlationIdOffset, correlationIdLength,
               0, false, null);
    }
    
    public void store(long index, int operation, byte[] topic,
            long topicOffset, long topicLength,
            byte[] data, long dataOffset, long dataLength,
            byte[] correlationId, long correlationIdOffset, long correlationIdLength,
            int expiration) throws StoreException
    {
        _store(index, operation, topic, topicOffset, topicLength,
               data, dataOffset, dataLength,
               correlationId, correlationIdOffset, correlationIdLength,
               expiration, true, null);
    }
    public void store(long index, int operation, byte[] topic,
            long topicOffset, long topicLength,
            byte[] data, long dataOffset, long dataLength,
            int flag, CommandId cmdId) throws StoreException
    {
        _store(index, operation, topic, topicOffset, topicLength,
            data, dataOffset, dataLength,
            null, 0, 0,
            flag, true, cmdId);
    }
    protected synchronized void _store(long index, int operation, byte[] topic,
                                   long topicOffset, long topicLength,
                                   byte[] data, long dataOffset, long dataLength,
                                   byte[] correlationId, long correlationIdOffset, long correlationIdLength,
                                   int expiration, boolean expirationIsValid, CommandId commandId) throws StoreException
    {
        Block current, next, first;
        
        long topicRemaining = topicLength;
        long dataRemaining = dataLength;
        long totalRemaining = topicLength + Block.CHAIN_HEADER + dataLength;
        boolean hasExtendedMetadata = correlationIdLength>0 || expirationIsValid || commandId != null;
        if(hasExtendedMetadata)
        {
            totalRemaining += 12 + correlationIdLength;
            if(commandId!=null) totalRemaining += commandId.id.length;
            
        }
        
        int lastBlockLength = (int)(totalRemaining % Block.BLOCK_DATA_SIZE);
        int blocksToWrite = (int)(totalRemaining / Block.BLOCK_DATA_SIZE) + (lastBlockLength>0? 1: 0);
        
        long crcVal = 0;
        if (_crc != null) /* we keep CRC off if we don't write to disk. */
        {
            _crc.reset();
            _crc.update(topic, (int) topicOffset, (int) topicLength);
            _crc.update(data, (int) dataOffset, (int) dataLength);
            crcVal = _crc.getValue();
        }
        try
        {
            first = next = get();
            if(first == null)
            {
                throw new StoreException("The store is full, and no blocks can be allocated.");
            }
            first.sequenceNumber = index;
            while(blocksToWrite > 0)
            {
                current = next;
                int bytesRemainingInBlock = Block.BLOCK_DATA_SIZE;
                if(blocksToWrite > 1)
                {
                    next = get();
                    if(next == null)
                    {
                        throw new StoreException("The store is full, and no additional blocks can be allocated.");
                    }
                } else
                {
                    next = null;
                }
       
                current.nextInChain = next;
                current.nextInList = null;
                
                // write the block info
                Buffer buffer = _sb == null ? _buffer : _sb; 
                long start = _sb == null ? current.getPosition() : 0;
                buffer.setPosition(start);
                if(current == first)
                {
                    buffer.putInt((int)totalRemaining);
                    
                }
                else
                {
                    buffer.putInt(next == null ? lastBlockLength : Block.BLOCK_DATA_SIZE);
                }
                buffer.putInt(next == null ? current.index : next.index);
                buffer.putLong(current==first?index:0);
                buffer.putLong(crcVal);
                
                // now write data and topic and stuff.
                buffer.setPosition(start + Block.BLOCK_HEADER_SIZE);
                if(current == first)
                {
                    // Adding bit for 4 to signify that there is an expiration
                    // and remain able to read previous versions of Stores
                    int commandIdLength = commandId!=null?commandId.id.length:0;
                    if(hasExtendedMetadata)
                    {
                        buffer.putByte((byte) ((operation & 0xff) | HAS_EXTENDED_METADATA));
                        buffer.putInt((int)topicLength);
                        buffer.putInt((int)expiration);
                        bytesRemainingInBlock -= (Block.CHAIN_HEADER + 4);
                        buffer.putInt(commandIdLength);
                        bytesRemainingInBlock -= 4;
                        if(commandIdLength>0)
                        {
                            _commandId.array = commandId.id;
                            _commandId.offset = 0;
                            _commandId.len = commandIdLength;
                            buffer.putBytes(_commandId);
                            bytesRemainingInBlock -= commandIdLength;
                        }
                        buffer.putInt((int)correlationIdLength); bytesRemainingInBlock -= 4;
                        if(correlationIdLength>0)
                        {
                            _argument.array = correlationId;
                            _argument.offset = correlationIdOffset;
                            _argument.len = correlationIdLength;
                            buffer.putBytes(_argument);
                            bytesRemainingInBlock -= correlationIdLength;
                        }
                        
                    }
                    else
                    {
                        buffer.putByte((byte) (operation & 0xff));
                        buffer.putInt((int)topicLength);
                        bytesRemainingInBlock -= Block.CHAIN_HEADER;
                    }
                    
                }
                long topicBytesToWrite = topicRemaining <= bytesRemainingInBlock ? topicRemaining : bytesRemainingInBlock;
                if(topicBytesToWrite > 0)
                {
                    _argument.array = topic;
                    _argument.offset = topicOffset + (topicLength - topicRemaining);
                    _argument.len = topicBytesToWrite;
                    buffer.putBytes(_argument);
                    bytesRemainingInBlock -= topicBytesToWrite;
                    topicRemaining -= topicBytesToWrite;
                
                }
                if(topicRemaining == 0)
                {
                    long dataBytesToWrite = dataRemaining <= bytesRemainingInBlock ? dataRemaining : bytesRemainingInBlock;
                    assert(dataBytesToWrite !=0 ); // if so, there's a math error -- we have more blocks allocated than we need.
                    _argument.array = data;
                    _argument.offset = dataOffset + (dataLength - dataRemaining);
                    _argument.len = dataBytesToWrite;
                    buffer.putBytes(_argument);
                    bytesRemainingInBlock -= dataBytesToWrite;
                    dataRemaining -= dataBytesToWrite;
                }
                // If we've used sb as an assembly area, here's where we actually write to the file.
                if(_sb != null)
                {
                    _buffer.setPosition(current.getPosition());
                    _argument.array = _sb.getBuffer();
                    _argument.offset = 0;
                    _argument.len = _sb.getPosition();
                    _buffer.putBytes(_argument);
                }
                --blocksToWrite;
                        
            }
            // now, there should be no data or topic left to write.
            assert(dataRemaining  == 0 && topicRemaining == 0);
            first.nextInList = null;
            if(_endOfUsedList != null)
            {
                _endOfUsedList.nextInList = first;
                _endOfUsedList = first;
            }
            else
            {
                // _endOfUsedList is null if _usedList is null, as well.
                _endOfUsedList = _usedList = first;
            }
            ++_usedCount;
            
        } catch (IOException e)
        {
            throw new StoreException(e);
        }
    }
        
    public void discardUpTo(long index) throws StoreException
    {
        // First, find the beginning of where we should discard.
        Block almostFreeList = detach(index);

        // we no longer need to keep a lock.  The only funny thing about our state is that
        // there are "missing" entries -- some on _usedList, some on _freeList, and some
        // on our (local) almostFreeList, about to be freed.  We could be fancy
        // and enqueue them for freeing, not going to do that.
        
        // For each one of these guys on our list, flatten onto the
        // new "zeroedOutList" and remember the last entry.  We'll use
        // that to efficiently prepend, below.
        Block zeroedOutList = null;
        Block lastOnTheList = almostFreeList;
        try
        {
            // Clear out the muck in the byte buffer
            Block current = almostFreeList;
            
            while(current != null)
            {
                Block next = current.nextInList;
                
                Block currentInChain = current;
                while(currentInChain != null)
                {
                    Block nextInChain = currentInChain.nextInChain;
                    _buffer.zero(currentInChain.getPosition(), Block.BLOCK_HEADER_SIZE);
                    currentInChain.sequenceNumber = 0;
                    currentInChain.nextInChain = null;
                    // and, flatten this chain onto the zeroed out list.
                    currentInChain.nextInList = zeroedOutList;
                    zeroedOutList = currentInChain;
                    
                    currentInChain = nextInChain;
                    
                }
                current = next;
            }
        } catch (IOException ioex)
        {
            throw new StoreException(ioex);
        }
        
        // now zeroedOutList is flat and free.  We can just prepend it to the free list.
        synchronized(this)
        {
            if(lastOnTheList != null)
            {
                lastOnTheList.nextInList = _freeList;
                _freeList = zeroedOutList;
            }
        }
    }

    public long getLowestUnpersisted()
    {
        if (_usedList != null)
        {
            return _usedList.sequenceNumber;
        }
        return -1;
    }

    private Block detach(long index)
    {
        Block keepList = null, endOfDiscardList = null, discardList = null;
        synchronized(this)
        {
            // advance through the list, find the first one to keep.  
            // we keep everything after that point, and discard everything up to that point.
            keepList = _usedList;
            while(keepList != null && keepList.sequenceNumber <= index)
            {
                if(discardList == null)
                {
                    discardList = keepList;
                }
                endOfDiscardList = keepList;
                keepList = keepList.nextInList;
                --_usedCount;
            }
            // detach the remainder of the list, and make it the new usedList.
            if(endOfDiscardList != null)
            {
                endOfDiscardList.nextInList = null;
                
            }
            _usedList = keepList;
            if(keepList == null)
            {
                _endOfUsedList = null;
            }
            this.notifyAll();
        }
        return discardList;
    }

    /**
     * Replays a single message onto a store replayer.
     * @param b  The first block of the message.  
     * @param replayer  The replayer to play this message on
     * @throws IOException
     */
    private void replayOnto(Block b, StoreReplayer replayer) throws IOException, DisconnectedException
    {
        _buffer.setPosition(b.getPosition());
        int totalLength = _buffer.getInt()- Block.CHAIN_HEADER;
        
        if(totalLength <=0)
        {
        	return; // this can occur when replaying during a disconnect scenario.
        }
        // don't care about the next in chain -- we already read that.
        _buffer.getInt();
        long index = _buffer.getLong();
        long crcVal = _buffer.getLong();
        
        _buffer.setPosition(b.getPosition() + Block.BLOCK_HEADER_SIZE);
        
        // grab the operation code, topic length that are guaranteed to be on the 1st block.
        int operation = _buffer.getByte();
        int topicLength = _buffer.getInt();
        
        int expiration = -1;
        CommandId cmdId = null;
        // if op & 4 then there is an expiration or it's a sow_delete
        if (((byte)operation & HAS_EXTENDED_METADATA) > 0)
        {
            expiration = _buffer.getInt();
            operation &= ~HAS_EXTENDED_METADATA;
            totalLength -= 4;
            _commandId.len = _buffer.getInt();
            totalLength -= 4;
            // grab the command id
            _buffer.getBytes(_commandId);
            cmdId = new CommandId(_commandId.array, (int)_commandId.offset, (int)_commandId.len);
            totalLength -= (int)_commandId.len;
            // grab the correlation id
            _correlationId.len = _buffer.getInt();
            totalLength -= 4;
            _buffer.getBytes(_correlationId);
            totalLength -= (int)_correlationId.len;
        }

        int expectedDataLength = totalLength - topicLength;
        if(b.nextInChain == null)
        {
            // everything fit into one block.  replay right out of the block memory.
            _argument.len = totalLength;
            _buffer.getBytes(_argument);
            if (_crc != null)
            {
                _crc.reset();
                try
                {
                    _crc.update(_argument.array, (int) _argument.offset,
                            topicLength);
                    _crc.update(_argument.array, (int) _argument.offset
                            + topicLength, expectedDataLength);
                } catch (Exception e)
                {
                    e.printStackTrace();
                }
                if (_crc.getValue() != crcVal)
                {
                    String message = String
                            .format("Corrupted message found.  Block index %d, sequence %d, topic length %d, expected data length %d, expected CRC %d, actual CRC %d",
                                    b.index, b.sequenceNumber, topicLength,
                                    expectedDataLength, crcVal, _crc.getValue());
                    throw new IOException(message);
                }
            }
            if (expiration == -1)
            {
                replayer.execute(index, operation, _argument.array,
                                 _argument.offset, topicLength, _argument.array,
                                 _argument.offset + topicLength, expectedDataLength,
                                 _correlationId.array, _correlationId.offset, _correlationId.len);
            }
            else if (operation == Message.Command.Publish ||
                     operation == Message.Command.DeltaPublish)
            {
                replayer.execute(index, operation, _argument.array,
                                 _argument.offset, topicLength, _argument.array,
                                 _argument.offset + topicLength,
                                 expectedDataLength, _correlationId.array, _correlationId.offset,
                                 _correlationId.len, expiration);
            }
            else // sow_delete
            {
                replayer.execute(index, operation, _argument.array,
                                 _argument.offset, topicLength, _argument.array,
                                 _argument.offset + topicLength,
                                 expectedDataLength, expiration, cmdId);
            }
        } else
        {
            // Discontinuous data.  we'll do this the hard way.
            // allocate enough space to hold it contiguously.  then, copy the pieces
            // in place.  then call up the replayer.
            if(_readBuffer == null || _readBuffer.length < totalLength)
            {
                _readBuffer = new byte[totalLength];
            }
            int offset =0;
            for(Block part = b; part != null; part = part.nextInChain)
            {
                if(b!=part) { _buffer.setPosition(part.getPosition() + Block.BLOCK_HEADER_SIZE); }
                int maxReadable = Block.BLOCK_DATA_SIZE;
                if(part == b) { maxReadable -= Block.CHAIN_HEADER; }
                
                int readLen = totalLength - offset > maxReadable ? maxReadable: totalLength-offset;
                _argument.len = readLen;
                try
                {
                    _buffer.getBytes(_argument);
                } catch(Exception e)
                {
                    e.printStackTrace();
                }
                System.arraycopy(_argument.array, (int)_argument.offset, _readBuffer, offset, readLen );
                offset += readLen;
            }
            if(offset != totalLength)
            {
                throw new IOException(String.format("Corrupted message found.  Expected %d bytes, read %d bytes.", totalLength, offset));
            }
            if (_crc != null)
            {
                _crc.reset();
                try
                {
                    _crc.update(_readBuffer, 0, topicLength);

                    _crc.update(_readBuffer, topicLength, offset - topicLength);
                } catch (Exception e)
                {
                    e.printStackTrace();
                }
                if (_crc.getValue() != crcVal)
                {
                    StringBuilder sb = new StringBuilder();
                    for (Block part = b; part != null; part = part.nextInChain)
                    {
                        sb.append(part.toString());
                        sb.append('\n');
                    }
                    throw new IOException(
                            String.format(
                                    "Corrupted multi-block message found in store. Topic Length %d, Data Length %d, BlockList:%n",
                                    topicLength, offset - topicLength)
                                    + sb.toString());

                }
            }
            if (expiration == -1)
            {
                replayer.execute(index, operation, _readBuffer, 0, topicLength,
                                 _readBuffer, topicLength, offset-topicLength,
                                 _correlationId.array, _correlationId.offset, _correlationId.len);
            }
            else if (operation == Message.Command.Publish ||
                     operation == Message.Command.DeltaPublish)
            {
                replayer.execute(index, operation, _readBuffer, 0, topicLength,
                                 _readBuffer, topicLength, offset-topicLength,
                                 _correlationId.array, _correlationId.offset, _correlationId.len,
                                 expiration);
            }
            else // sow_delete
            {
                replayer.execute(index, operation, _readBuffer, 0, topicLength,
                                 _readBuffer, topicLength, offset-topicLength,
                                 expiration, cmdId);
            }
        }
    }
    public synchronized void replay(StoreReplayer replayer) throws StoreException, DisconnectedException
    {
        // for each used block
        try
        {
            
            for(Block b = _usedList; b!=null; b=b.nextInList)
            {
                replayOnto(b, replayer);
            }
            
        }
        catch(IOException ioex)
        {
            throw new StoreException(ioex);
        }
    }

    public synchronized void replaySingle(StoreReplayer replayer, long index)
            throws StoreException, DisconnectedException
    {
        try
        {
            // find the first one that equals this index.
        
            for(Block b = this._usedList; b != null; b = b.nextInList)
            {
                if(b.sequenceNumber == index)
                {
                    replayOnto(b, replayer);
                    return;
                }
            }
        }
        catch (IOException ioex)
        {
            throw new StoreException(ioex);
        }
    }

    public long unpersistedCount()
    {
    	return _usedCount;
    }
    
    Block findOrCreate(long index, HashMap<Long, Block> allBlocks)
    {
        if(allBlocks.containsKey(index))
        {
            return allBlocks.get(index);
        }
        else
        {
            Block b= new Block((int)index);
            allBlocks.put(index, b);
            return b;
        }
    }
    
    protected synchronized void recover() throws StoreException
    {
        _usedList = _freeList = null;
        _usedCount = 0;
        
        // Start here and work the buffer.
        HashMap<Long, Block> allBlocks = new HashMap<Long, Block>();   // file index -> block
        TreeMap<Long, Block> heads = new TreeMap<Long, Block>();       // sequence number --> block
        try
        {
            long end = _buffer.getSize();
            for(long offset = 0; offset < end; offset+=Block.SIZE)
            {
                long index = offset/Block.SIZE;
                Block b = findOrCreate(index, allBlocks);
                _buffer.setPosition(offset);
                int size = _buffer.getInt();
                int nextIndex = _buffer.getInt();
                b.nextInChain = size == 0 || nextIndex == index ? null : findOrCreate(nextIndex, allBlocks);

                long sequence = _buffer.getLong();
                
                // All empty blocks are free. That's the rule.
                if(size == 0)
                {
                    b.nextInList = _freeList;
                    _freeList = b;
                }
                else if(sequence != 0)
                {
                    // Only head blocks have their sequence number set.
                    b.sequenceNumber = sequence;
                    heads.put(sequence, b);
                }
                // else, Belongs to part of a "chain"
            }
            // Need to put the heads onto the list in ascending order.
            for(Entry<Long, Block> e: heads.entrySet())
            {
                Block b = e.getValue();
                b.nextInList = null;
                if(_endOfUsedList != null)
                {
                    _endOfUsedList.nextInList = b;
                    _endOfUsedList = b;
                }
                else
                {
                    _usedList = _endOfUsedList = b;
                }
                ++_usedCount;
            }
        }
        catch(IOException ioex)
        {
            throw new StoreException(ioex);
        }
    }

    protected void growFreeListIfEmpty() throws StoreException
    {
        if (_freeList != null) return;
        try {
            long oldSize = _buffer.getSize();
            long newSize = oldSize + ((long)Block.SIZE * (long)_blocksPerRealloc);
            if (newSize > Integer.MAX_VALUE)
            {
                throw new StoreException("PublishStore capacity exceeded");
            }
            while (_buffer.getSize() == oldSize)
            {
                _buffer.setSize(newSize);
                // Check to make sure resizeHandler didn't block request without
                // freeing up room before proceeding
                if (_freeList != null) return;
            }
            if (_buffer.getSize() < newSize)
                throw new StoreException("Publish store could not resize, possibly due to resize handler.");
            for (int idx = (int) (oldSize / Block.SIZE); idx < (int) (newSize / Block.SIZE); idx++)
            {
                Block b = new Block(idx);
                _buffer.zero(b.getPosition(), Block.SIZE);
                b.nextInChain = null;
                b.nextInList = _freeList;
                _freeList = b;
            }
        } catch (IOException e)
        {
            throw new StoreException(e);
        }
    }

    public synchronized void flush() throws TimedOutException
    {
        if (_usedList == null) return;
        long current = _endOfUsedList.sequenceNumber;
        while (_usedList != null && current >= _usedList.sequenceNumber)
        {
            try
            {
                this.wait();
            } catch (InterruptedException e)
            {
            }
        }
    }

    public synchronized void flush(long timeout) throws TimedOutException
    {
        if (_usedList == null) return;
        long current = _endOfUsedList.sequenceNumber;
        long start = System.currentTimeMillis();
        long end = System.currentTimeMillis();
        while (_usedList != null && current >= _usedList.sequenceNumber && (end - start) < timeout)
        {
            try
            {
                long remaining = timeout - (end - start);
                this.wait(remaining);
                end = System.currentTimeMillis();
                if ((end - start) >= timeout)
                    throw new TimedOutException("Timed out waiting to flush publish store.");
            } catch (InterruptedException e)
            {
            }
        }
        // Check if we exited loop due to timeout
        if (_usedList != null && current >= _usedList.sequenceNumber && (end - start) >= timeout)
        {
            throw new TimedOutException("Timed out waiting to flush publish store.");
        }
    }

    public void setResizeHandler(PublishStoreResizeHandler handler)
    {
        _buffer.setResizeHandler(handler, this);
    }
        
    /********* PRIVATE METHODS *************/

	private Block get() throws StoreException
    {
        growFreeListIfEmpty();
        Block b = _freeList;
        if(b != null)
        {
            _freeList = b.nextInList;
        }
        return b;
    }

}
