package com.crankuptheamps.examples;

import com.crankuptheamps.client.BookmarkStore;
import com.crankuptheamps.client.MemoryBookmarkStore;
import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.Message;
import com.crankuptheamps.client.MessageStream;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;
import com.crankuptheamps.client.exception.AMPSException;

import java.lang.Thread;
import java.lang.Thread.State;

import java.io.Closeable;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.HashSet;

import java.util.regex.*;

class AMPSBookmarkStore
      extends MemoryBookmarkStore
      implements BookmarkStore, Closeable
{

  /**
   *  Create a bookmark store using an AMPS instance for persistent storage.
   *  To use this class, you must provide a client that is connected to the AMPS
   *  instance to be used for storage, and the name of the client to maintain
   *  state for.
   *
   *  @param bookmarkClient The client to use to manage the bookmark store. This should be a
   *  different object than the client that owns the subscriptions, even when the
   *  bookmark store is stored in the same AMPS instance.
   *  @param trackedClientName the name of the client to track state for
   */

   public AMPSBookmarkStore(Client bookmarkClient, String trackedClientName)
          throws AMPSException
   {
      if (bookmarkClient == null)
      {
        throw new AMPSException("Null client passed to bookmark store.");
      }

      _internalClient = bookmarkClient;
      _trackedName = trackedClientName;

      _bookmarkPattern =     Pattern.compile("\"bookmark\" *: *\"([^\"]*)\"");
      _subIdPattern =        Pattern.compile("\"subId\" *: *\"([^\"]+)\"");
      _persistedAckPattern = Pattern.compile("\"persisted\" *: *\"([^\"]+)\"");

      MessageStream ms = null;

      try
      {
        // Message to use for logging bookmarks
        Message logmsg = _internalClient.allocateMessage();

        // Retrieve state for this client.
        ms = _internalClient.sow("/ADMIN/bookmark_store",
                                 "/clientName = '"
                                 + _trackedName + "'");
         for (Message msg : ms)
         {
            if (msg.getCommand() != Message.Command.SOW )
            {  continue; }

            String data = msg.getData();

            Matcher bookmarkMatch = _bookmarkPattern.matcher(data);
            if(! bookmarkMatch.find()) continue;

            Matcher subIdMatch = _subIdPattern.matcher(data);
            if (! subIdMatch.find()) continue;

            Matcher persistedAckMatch = _persistedAckPattern.matcher(data);
            if (! persistedAckMatch.find()) continue;

            // Get the bookmark string, the subId, and whether this subscription
            // receives persisted acks.
            String bookmark = bookmarkMatch.group(1);
            String subId = subIdMatch.group(1);
            String persistedAcks = persistedAckMatch.group(1);

            // Extract individual bookmarks from the record if necessary
            String[] bookmarks = new String[1]; 
            if (bookmark.contains(","))
            {
               bookmarks = bookmark.split(",");
            }
            else
            {
               bookmarks[0] = bookmark;
            }

            for (String b : bookmarks)
            {
              // Create a message with the subId and bookmark
              // to use for logging with the MemoryBookmarkStore.
              logmsg.reset();
              logmsg.setSubId(subId);
              logmsg.setBookmark(bookmark);

              // Register the bookmark from the SOW as the 
              // last successfully processed bookmark for this subId.
              super.log(logmsg);
              super.discard(logmsg);

              if (persistedAcks == "true")
              {
                 super.persisted(logmsg.getSubIdRaw(), logmsg.getBookmarkRaw());
                 _persistedAcks.add(logmsg.getSubId()); 
              } 
            }
            _discardCounter.put(logmsg.getSubIdRaw(), 0); 
         }
       }
       catch (AMPSException e)
       {
           System.err.println(e.getLocalizedMessage());
           e.printStackTrace(System.err);
           throw e;
       }
       finally { if (ms != null) ms.close(); }

       // Start the worker to asynchronously handle updates
       _workerThread = new Thread(
                          new UpdatePublisher(_internalClient,
                                              _trackedName,
                                              _workQueue),
                          "Bookmark Update for " + _trackedName);
       _workerThread.start();
   }
  
   /**
    * Discard the specified message.
    * Call discard() on the underlying MemoryBookmarkStore and 
    * check to see if we need to persist the state of this
    * subId. 
    * @param subId the subscription from which to discard the message
    * @param bookmarkSeqNo the sequence number the store has assigned to the bookmark to be discarded
    * @deprecated 
    */
   @Override
   public void discard(Field subId, long bookmarkSeqNo) throws AMPSException
   {
       super.discard(subId, bookmarkSeqNo);
       checkForPersist(subId);
   }


   /**
    * Discard the specified message.
    * Call discard() on the underlying MemoryBookmarkStore and 
    * check to see if we need to persist the state of this
    * subId. 
    * @param m the message to discard
    */ 
   @Override
   public void discard(Message m) throws AMPSException
   {
     super.discard(m);
     checkForPersist(m.getSubIdRaw()); 
   }

   /**
    * Handle persisted acks.
    * For bookmark live subscriptions, persisted acks indicate the point at which the message
    * stream has been fully persisted and can be reliably recovered.
    * @param subId the subscription id which has received the ack
    * @param bookmark the bookmark that has been persisted
    */
    @Override
    public void persisted(Field subId, BookmarkField bookmark) throws AMPSException
    {
       super.persisted(subId, bookmark);
       _persistedAcks.add(subId.toString()); 
    }

   /**
    * Handle persisted acks.
    * For bookmark live subscriptions, persisted acks indicate the point at which the message
    * stream has been fully persisted and can be reliably recovered.
    * @param subId the subscription id which has received the ack
    * @param bookmark the bookmark that has been persisted
    */
    @Override
    public void persisted(Field subId, long bookmark) throws AMPSException
    {
       super.persisted(subId, bookmark);
       _persistedAcks.add(subId.toString()); 
    }

   /** Implement close().
    *  Persists current state of all saved bookmarks, then
    *  closes the client used for administrative updates.
    */

   public void close()
   {
     for (Field subId : _discardCounter.keySet())
     {
        try
        {
          _workQueue.put(new UpdateRecord(subId.copy(), getMostRecent(subId)));
        }
        catch (AMPSException e)
        {
           e.printStackTrace();
           // Swallow exception: could also translate to unchecked
           // or log in the object.
        }
        catch (InterruptedException e)
        {
          // Unable to update publish store. Recover at last saved bookmark
          // instead.
          e.printStackTrace();
          break;
        }
     }
     // Wait for the worker thread to drain the queue.
     while (!(_workQueue.size() == 0 &&
              _workerThread.getState() == Thread.State.WAITING))
     {
        Thread.yield();
     }
     _workerThread.interrupt();
     try
     {
        _workerThread.join();
     }
     catch (InterruptedException e)
     {
        e.printStackTrace();
        // trying to close, continue
     }
     _internalClient.close();
     _internalClient = null;
   }

   /**
    * Check to see if it's time to persist a message to the
    * AMPS store. If so, queue the update. 
    * @param subId the subscription ID to check
    */

   private void checkForPersist(Field subId)  throws AMPSException
   {
     Integer count = _discardCounter.get(subId);
     if (count == null) { count = 0; }
     ++count;
     if (count > _threshold)
     {
          try
          {
             _workQueue.put(new UpdateRecord(subId.copy(), getMostRecent(subId)));
             count = 0;
          }
          catch (InterruptedException e)
          {
             throw new AMPSException(e);
          }
     }    
     _discardCounter.put(subId, count);
   }

   /**
     * Set how often to persist the bookmark state to AMPS.
     * @param messageCount the persistence interval, in number of messages processed
     */
   public void setPersistEvery(int messageCount)
   {
      _threshold = messageCount;
   }

   private Client _internalClient;
   private String _trackedName;
   private Pattern _bookmarkPattern;
   private Pattern _subIdPattern;
   private Pattern _persistedAckPattern;
   private final ConcurrentHashMap<Field, Integer> _discardCounter = new ConcurrentHashMap<Field, Integer>(100); 
   private final Set<String> _persistedAcks = new HashSet<>();
   private int _threshold = 1000;
   private BlockingQueue<UpdateRecord> _workQueue =
           new ArrayBlockingQueue<UpdateRecord>(10);
   private Thread _workerThread;
   private boolean _getsPersistedAcks = false;

   /**
     * Internal class for providing update information to the worker thread.
     */

   private class UpdateRecord
   {
      public Field subId;
      public Field bookmark;
      public UpdateRecord(Field subId_, Field bookmark_)
      {
           subId = subId_;
           bookmark = bookmark_; 
      }
   }

   /**
     *  Internal class for updating state.
     */
   private class UpdatePublisher implements Runnable
   {
         private BlockingQueue<UpdateRecord> _workQueue;
         private Client _internalClient;
         private String _trackedName;

         public UpdatePublisher(Client client,
                                String trackedName,
                                BlockingQueue<UpdateRecord> workQueue)
         {
             _workQueue = workQueue;
             _trackedName = trackedName;
             _internalClient = client; 
         }

        /**
          * Entry point for the runnable. Retrieves the next UpdateRecord
          * from the queue and publishes that information to AMPS.
          */
        public void run()
        {
            while(true)
            { 
               try
               {

                 UpdateRecord update = _workQueue.take();
                 String msg = "{\"clientName\":\"" + _trackedName + "\""
                            + ",\"subId\":\"" + update.subId + "\""
                            + ",\"bookmark\":\"" + update.bookmark + "\""
                            + ",\"persisted\":" + "\"" + _persistedAcks.contains(update.subId.toString()) + "\""
                            + "}";
                   _internalClient.publish("/ADMIN/bookmark_store", msg);
               }
               // For sample purposes, print errors and keep going. For production,
               // recovery might include notifying the store or the application that
               // an error has occured.
               catch (AMPSException e)
               {
                  System.out.println("Unable to write update: " );
                  e.printStackTrace(); 
                  return;
               }
               catch (InterruptedException e)
               {
                  // This is only an error if there are unprocessed records
                  if (_workQueue.size() != 0 ) e.printStackTrace();
                  return;
               }
             
            }
        }
   }  
     
}

